#include "journalviewmodel.hpp"
#include "journalindex.hpp"

#include <fstream>
#include <map>

#include <MyGUI_LanguageManager.h>

#include <components/misc/strings/algorithm.hpp>
#include <components/translation/translation.hpp>

#include "../mwbase/environment.hpp"
#include "../mwbase/journal.hpp"
#include "../mwbase/windowmanager.hpp"
#include "../mwbase/world.hpp"

#include "../mwdialogue/keywordsearch.hpp"
#include "../mwworld/datetimemanager.hpp"

namespace MWGui
{
    namespace
    {
        bool isAsciiAlpha(Utf8Stream::UnicodeChar character)
        {
            return (character >= 'A' && character <= 'Z') || (character >= 'a' && character <= 'z');
        }

        Utf8Stream::UnicodeChar toHiragana(Utf8Stream::UnicodeChar character)
        {
            if (character >= 0x30A1 && character <= 0x30F6)
                return character - 0x60;
            return character;
        }

        Utf8Stream::UnicodeChar classifyJapaneseKanaIndex(Utf8Stream::UnicodeChar character)
        {
            const Utf8Stream::UnicodeChar ch = toHiragana(character);

            if ((ch >= 0x3041 && ch <= 0x304A) || ch == 0x3094)
                return JournalIndex::sJapaneseA;
            if (ch >= 0x304B && ch <= 0x3054)
                return JournalIndex::sJapaneseKa;
            if (ch >= 0x3055 && ch <= 0x305E)
                return JournalIndex::sJapaneseSa;
            if (ch >= 0x305F && ch <= 0x3069)
                return JournalIndex::sJapaneseTa;
            if (ch >= 0x306A && ch <= 0x306E)
                return JournalIndex::sJapaneseNa;
            if (ch >= 0x306F && ch <= 0x307D)
                return JournalIndex::sJapaneseHa;
            if (ch >= 0x307E && ch <= 0x3082)
                return JournalIndex::sJapaneseMa;
            if (ch >= 0x3083 && ch <= 0x3088)
                return JournalIndex::sJapaneseYa;
            if (ch >= 0x3089 && ch <= 0x308D)
                return JournalIndex::sJapaneseRa;
            if (ch >= 0x308E && ch <= 0x3096)
                return JournalIndex::sJapaneseWa;

            return 0;
        }

        bool isSkippableLeadingCharacter(Utf8Stream::UnicodeChar character)
        {
            switch (character)
            {
                case 0x0009: // \t
                case 0x0020: // space
                case 0x0022: // "
                case 0x0027: // '
                case 0x0028: // (
                case 0x0029: // )
                case 0x005B: // [
                case 0x005D: // ]
                case 0x30FB: // ・
                case 0x3000: // ideographic space
                case 0x300C: // 「
                case 0x300D: // 」
                case 0x300E: // 『
                case 0x300F: // 』
                case 0xFF08: // full-width (
                case 0xFF09: // full-width )
                    return true;
                default:
                    return false;
            }
        }

        Utf8Stream::UnicodeChar firstSignificantCharacter(std::string_view text)
        {
            Utf8Stream stream(text);
            while (!stream.eof())
            {
                Utf8Stream::UnicodeChar character = stream.peek();
                stream.consume();

                if (character == Utf8Stream::sBadChar() || isSkippableLeadingCharacter(character))
                    continue;

                return character;
            }
            return 0;
        }

        const std::map<std::string, std::string, std::less<>>& japaneseTopicReadings()
        {
            static const auto dictionary = []() {
                std::map<std::string, std::string, std::less<>> result;

                std::ifstream stream("journal_japanese_yomi.tsv");
                if (!stream.is_open())
                    return result;

                std::string line;
                while (std::getline(stream, line))
                {
                    if (!line.empty() && line.back() == '\r')
                        line.pop_back();
                    if (line.empty() || line[0] == '#')
                        continue;

                    const size_t tab = line.find('\t');
                    if (tab == std::string::npos || tab == 0 || tab + 1 >= line.size())
                        continue;

                    result.emplace(line.substr(0, tab), line.substr(tab + 1));
                }

                return result;
            }();

            return dictionary;
        }

        Utf8Stream::UnicodeChar classifyJapaneseTopic(std::string_view topicName)
        {
            Utf8Stream::UnicodeChar first = firstSignificantCharacter(topicName);

            if (isAsciiAlpha(first))
                return JournalIndex::sJapaneseEnglish;

            if (Utf8Stream::UnicodeChar index = classifyJapaneseKanaIndex(first))
                return index;

            const auto& dictionary = japaneseTopicReadings();
            const auto it = dictionary.find(topicName);
            if (it != dictionary.end())
            {
                const Utf8Stream::UnicodeChar readingFirst = firstSignificantCharacter(it->second);
                if (isAsciiAlpha(readingFirst))
                    return JournalIndex::sJapaneseEnglish;

                if (Utf8Stream::UnicodeChar index = classifyJapaneseKanaIndex(readingFirst))
                    return index;
            }

            // Fallback bucket when no yomi is available.
            return JournalIndex::sJapaneseWa;
        }
    }

    struct JournalViewModelImpl;

    struct JournalViewModelImpl : JournalViewModel
    {
        using TopicSearch = MWDialogue::KeywordSearch<const MWDialogue::Topic*>;

        mutable bool mKeywordSearchLoaded;
        mutable TopicSearch mKeywordSearch;

        JournalViewModelImpl() { mKeywordSearchLoaded = false; }

        virtual ~JournalViewModelImpl() = default;

        void load() override {}

        void unload() override
        {
            mKeywordSearch.clear();
            mKeywordSearchLoaded = false;
        }

        void ensureKeyWordSearchLoaded() const
        {
            if (!mKeywordSearchLoaded)
            {
                MWBase::Journal* journal = MWBase::Environment::get().getJournal();

                for (const auto& [_, topic] : journal->getTopics())
                    mKeywordSearch.seed(topic.getName(), &topic);

                mKeywordSearchLoaded = true;
            }
        }

        bool isEmpty() const override
        {
            MWBase::Journal* journal = MWBase::Environment::get().getJournal();

            return journal->getEntries().empty();
        }

        template <typename EntryType, typename Interface>
        struct BaseEntry : Interface
        {
            const EntryType* mEntry;
            JournalViewModelImpl const* mModel;

            BaseEntry(JournalViewModelImpl const* model, const EntryType& entry)
                : mEntry(&entry)
                , mModel(model)
                , loaded(false)
            {
            }

            virtual ~BaseEntry() = default;

            mutable bool loaded;
            mutable std::string utf8text;

            // hyperlinks in @link# notation
            mutable std::map<std::pair<size_t, size_t>, const MWDialogue::Topic*> mHyperLinks;

            virtual std::string getText() const = 0;

            void ensureLoaded() const
            {
                if (!loaded)
                {
                    mModel->ensureKeyWordSearchLoaded();

                    utf8text = getText();

                    size_t posEnd = 0;
                    for (;;)
                    {
                        const size_t posBegin = utf8text.find('@');
                        if (posBegin != std::string::npos)
                            posEnd = utf8text.find('#', posBegin);

                        if (posBegin != std::string::npos && posEnd != std::string::npos)
                        {
                            std::string link = utf8text.substr(posBegin + 1, posEnd - posBegin - 1);
                            const char specialPseudoAsteriskCharacter = 127;
                            std::replace(link.begin(), link.end(), specialPseudoAsteriskCharacter, '*');
                            std::string_view topicName = MWBase::Environment::get()
                                                             .getWindowManager()
                                                             ->getTranslationDataStorage()
                                                             .topicStandardForm(link);

                            std::string displayName = link;
                            while (displayName[displayName.size() - 1] == '*')
                                displayName.erase(displayName.size() - 1, 1);

                            utf8text.replace(posBegin, posEnd + 1 - posBegin, displayName);

                            const MWDialogue::Topic* value = nullptr;
                            if (mModel->mKeywordSearch.containsKeyword(topicName, value))
                                mHyperLinks[std::make_pair(posBegin, posBegin + displayName.size())] = value;
                        }
                        else
                            break;
                    }

                    loaded = true;
                }
            }

            std::string_view body() const override
            {
                ensureLoaded();

                return utf8text;
            }

            void visitSpans(std::shared_ptr<BookTypesetter> mTypesetter,
                std::function<void(const MWDialogue::Topic*, size_t, size_t)> visitor) const override
            {
                ensureLoaded();
                mModel->ensureKeyWordSearchLoaded();

                if (mHyperLinks.size()
                    && MWBase::Environment::get().getWindowManager()->getTranslationDataStorage().hasTranslation())
                {
                    mTypesetter->addContent(body());

                    size_t formatted = 0; // points to the first character that is not laid out yet
                    for (const auto& [range, topicId] : mHyperLinks)
                    {
                        if (formatted < range.first)
                            visitor(0, formatted, range.first);
                        visitor(topicId, range.first, range.second);
                        formatted = range.second;
                    }
                    if (formatted < utf8text.size())
                        visitor(0, formatted, utf8text.size());
                }
                else
                {
                    std::vector<TopicSearch::Match> matches;
                    mModel->mKeywordSearch.highlightKeywords(utf8text.begin(), utf8text.end(), matches);

                    TopicSearch::removeUnusedPostfix(utf8text, matches);
                    mTypesetter->addContent(body());

                    std::string::const_iterator i = utf8text.begin();
                    for (const TopicSearch::Match& match : matches)
                    {
                        if (i != match.mBeg)
                            visitor(0, i - utf8text.begin(), match.mBeg - utf8text.begin());

                        visitor(match.mValue, match.mBeg - utf8text.begin(), match.mEnd - utf8text.begin());

                        i = match.mEnd;
                    }

                    if (i != utf8text.end())
                        visitor(0, i - utf8text.begin(), utf8text.size());
                }
            }
        };

        void visitQuestNames(bool activeOnly, std::function<void(std::string_view, bool)> visitor) const override
        {
            MWBase::Journal* journal = MWBase::Environment::get().getJournal();

            std::set<std::string_view, Misc::StringUtils::CiComp> visitedQuests;

            // Note that for purposes of the journal GUI, quests are identified by the name, not the ID, so several
            // different quest IDs can end up in the same quest log. A quest log should be considered finished
            // when any quest ID in that log is finished.
            for (const auto& [_, quest] : journal->getQuests())
            {
                // Unfortunately Morrowind.esm has no quest names, since the quest book was added with tribunal.
                // Note that even with Tribunal, some quests still don't have quest names. I'm assuming those are not
                // supposed to appear in the quest book.
                const std::string_view questName = quest.getName();
                if (questName.empty())
                    continue;
                // Don't list the same quest name twice
                if (!visitedQuests.insert(questName).second)
                    continue;

                bool isFinished = std::ranges::find_if(journal->getQuests(), [&](const auto& pair) {
                    return pair.second.isFinished() && Misc::StringUtils::ciEqual(questName, pair.second.getName());
                }) != journal->getQuests().end();

                if (activeOnly && isFinished)
                    continue;

                visitor(questName, isFinished);
            }
        }

        struct JournalEntryImpl : BaseEntry<MWDialogue::StampedJournalEntry, JournalEntry>
        {
            mutable std::string timestamp_buffer;

            JournalEntryImpl(JournalViewModelImpl const* model, const MWDialogue::StampedJournalEntry& entry)
                : BaseEntry(model, entry)
            {
            }

            std::string getText() const override { return mEntry->getText(); }

            std::string_view timestamp() const override
            {
                if (timestamp_buffer.empty())
                {
                    // std::string dayStr = MyGUI::LanguageManager::getInstance().replaceTags("#{sDay}");

                    std::ostringstream os;

                    os << MWBase::Environment::get().getWorld()->getTimeManager()->getMonthName(mEntry->mMonth)
                       << " " << mEntry->mDayOfMonth
                       << "日 (第" << mEntry->mDay << "天)";

                    timestamp_buffer = os.str();
                }

                return timestamp_buffer;
            }
        };

        void visitJournalEntries(
            std::string_view questName, std::function<void(JournalEntry const&, const MWDialogue::Quest*)> visitor) const override
        {
            MWBase::Journal* journal = MWBase::Environment::get().getJournal();

            // if (!questName.empty())
            {
                std::vector<MWDialogue::Quest const*> quests;
                for (const auto& [_, quest] : journal->getQuests())
                {
                    if (questName.empty() || Misc::StringUtils::ciEqual(quest.getName(), questName))
                        quests.push_back(&quest);
                }

                for (const MWDialogue::StampedJournalEntry& journalEntry : journal->getEntries())
                {
                    bool visited = false;
                    for (const MWDialogue::Quest* quest : quests)
                    {
                        if (quest->getTopic() != journalEntry.mTopic)
                            continue;
                        for (const MWDialogue::Entry& questEntry : *quest)
                        {
                            if (journalEntry.mInfoId == questEntry.mInfoId)
                            {
                                visitor(JournalEntryImpl(this, journalEntry),
                                    questName.empty() ? quest : nullptr);
                                visited = true;
                                break;
                            }
                        }
                    }
                    if (!visited && questName.empty())
                        visitor(JournalEntryImpl(this, journalEntry), nullptr);
                }
            }
            // else
            // {
            //     for (const MWDialogue::StampedJournalEntry& journalEntry : journal->getEntries())
            //         visitor(JournalEntryImpl(this, journalEntry));
            // }
        }

        void visitTopicName(
            const MWDialogue::Topic& topic, std::function<void(std::string_view)> visitor) const override
        {
            visitor(topic.getName());
        }

        void visitTopicNamesStartingWith(
            Utf8Stream::UnicodeChar character, std::function<void(std::string_view)> visitor) const override
        {
            MWBase::Journal* journal = MWBase::Environment::get().getJournal();

            if (JournalIndex::isJapaneseIndex(character))
            {
                for (const auto& [_, topic] : journal->getTopics())
                {
                    if (classifyJapaneseTopic(topic.getName()) == character)
                        visitor(topic.getName());
                }
                return;
            }

            character = Utf8Stream::toLowerUtf8(character);
            for (const auto& [_, topic] : journal->getTopics())
            {
                Utf8Stream stream(topic.getName());
                Utf8Stream::UnicodeChar first = Utf8Stream::toLowerUtf8(stream.peek());

                if (Translation::isFirstChar(first, (char)character))
                    visitor(topic.getName());
            }
        }

        struct TopicEntryImpl : BaseEntry<MWDialogue::Entry, TopicEntry>
        {
            MWDialogue::Topic const& mTopic;

            TopicEntryImpl(
                JournalViewModelImpl const* model, MWDialogue::Topic const& topic, const MWDialogue::Entry& entry)
                : BaseEntry(model, entry)
                , mTopic(topic)
            {
            }

            std::string getText() const override { return mEntry->getText(); }

            std::string_view source() const override { return mEntry->mActorName; }
        };

        void visitTopicEntries(
            const MWDialogue::Topic& topic, std::function<void(TopicEntry const&)> visitor) const override
        {
            for (const MWDialogue::Entry& entry : topic)
                visitor(TopicEntryImpl(this, topic, entry));
        }
    };

    std::shared_ptr<JournalViewModel> JournalViewModel::create()
    {
        return std::make_shared<JournalViewModelImpl>();
    }

}
